package gov.va.genisis2.dao.impl;

import gov.va.genisis2.common.enums.CommonEnum;
import gov.va.genisis2.common.enums.WorkflowStatusEnum;
import gov.va.genisis2.dao.IRequestDao;
import gov.va.genisis2.exceptions.ErrorEnum;
import gov.va.genisis2.exceptions.GenisisDAOException;
import gov.va.genisis2.model.Request;
import gov.va.genisis2.model.RequestHistory;
import gov.va.genisis2.model.WorkflowStatus;
import gov.va.genisis2.util.FortifyHelper;

import java.util.List;

import javax.persistence.EntityManagerFactory;
import javax.persistence.NoResultException;
import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;
import javax.persistence.criteria.Subquery;

import org.apache.commons.lang3.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.criterion.Projections;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

/**
 * The Class RequestDao.
 *
 * The Request data access object (DAO) is an object that provides an abstract
 * interface to some type of database or other persistence mechanism. By mapping
 * application calls to the persistence layer, Request DAO provide some specific
 * data operations without exposing details of the database.
 */
@Repository
@Transactional(value = "transactionManager")
public class RequestDao implements IRequestDao {

	/** The LOGGER. */
	private static final Logger LOGGER = LoggerFactory.getLogger(RequestDao.class);

	@Autowired
	private SessionFactory sessionFactory;

	/**
	 * This method is used to create request.
	 * 
	 * @param entity
	 *            The entity.
	 * @return int This returns created request id.
	 */
	@Override
	public int createRequest(final Request entity) throws GenisisDAOException {
		if (entity == null) {
			return 0;
		}
		Integer request = this.maxRowValue();
		if (request == 0) {
			if (LOGGER.isInfoEnabled()) {
				LOGGER.info("Exception occurred while getting Max value on Request table.Unable to create Request");
			}
			return 0;
		} else
			entity.setId(request.intValue());
		try {
			Session session = sessionFactory.getCurrentSession();
			request = (Integer) session.save(entity);
			session.flush();
		} catch (Exception ex) {
			LOGGER.error("Exception occurred on createRequest.", ex);
			throw new GenisisDAOException("Exception occurred on createRequest.", ex);
		}
		return request.intValue();
	}

	/**
	 * This method is used to get request by UID.
	 * 
	 * @param uid
	 *            The uid.
	 * @return this returns list of requestes.
	 */
	@Override
	public List<Request> getRequestsByUID(String uid) throws GenisisDAOException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("getRequestsByUID: " + FortifyHelper.getCleanStringInput(uid));
		}

		List<Request> listRequest = null;
		try {
			Session session = sessionFactory.getCurrentSession();
			CriteriaBuilder criteriaBuilder = session.getCriteriaBuilder();
			CriteriaQuery<Request> query = criteriaBuilder.createQuery(Request.class);
			Root<Request> root = query.from(Request.class);
			Predicate condition = criteriaBuilder.equal(root.get(CommonEnum.REQUEST_CREATED_BY.getText()), uid);
			query.where(condition);
			listRequest = session.createQuery(query).getResultList();
		} catch (Exception ex) {
			LOGGER.error("Exception occurred while querying  getStudyApprovalsByUID.", ex);
			throw new GenisisDAOException("Exception occurred while querying  getStudyApprovalsByUID.", ex);
		}
		return listRequest;
	}

	/**
	 * This method is used to getRequestsDataManagers.
	 * 
	 * @param uid
	 *            The uid.
	 * @return this returns list of requests data managers.
	 */
	@Override
	public List<Request> getRequestsDataManagers(String uid) throws GenisisDAOException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("getRequestsDataManagers: " + FortifyHelper.getCleanStringInput(uid));
		}

		List<Request> listRequest = null;
		try {
			Session session = sessionFactory.getCurrentSession();
			CriteriaBuilder criteriaBuilder = session.getCriteriaBuilder();
			CriteriaQuery<Request> query = criteriaBuilder.createQuery(Request.class);
			Root<Request> root = query.from(Request.class);
			Predicate condition1 = criteriaBuilder.notEqual(root.get(CommonEnum.REQUEST_CREATED_BY.getText()), uid);
			Predicate condition2 = criteriaBuilder.notEqual(root.get(CommonEnum.STATUS_DESC.getText()), WorkflowStatusEnum.DRAFT.getDesc());
			Predicate condition3 = criteriaBuilder.equal(root.get(CommonEnum.REQUEST_CREATED_BY.getText()), uid);
			Predicate condition4 = criteriaBuilder.and(condition1, condition2);
			Predicate conditions = criteriaBuilder.or(condition4, condition3);
			query.where(conditions);
			listRequest = session.createQuery(query).getResultList();
		} catch (Exception ex) {
			LOGGER.error("Exception occurred while querying  getRequestsDataManagers.", ex);
			throw new GenisisDAOException("Exception occurred while querying  getRequestsDataManagers.", ex);
		}
		return listRequest;
	}

	/**
	 * This method is used to getRequestsByDataManager.
	 * 
	 * @param uid
	 *            The uid.
	 * @return this returns list of requests.
	 */
	@Override
	public List<Request> getRequestsByDataManager(String uid) throws GenisisDAOException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("getRequestsByDataManager: " + uid);
		}

		List<Request> listRequest = null;
		try {
			Session session = sessionFactory.getCurrentSession();
			CriteriaBuilder criteriaBuilder = session.getCriteriaBuilder();
			CriteriaQuery<Request> query = criteriaBuilder.createQuery(Request.class);
			Root<Request> root = query.from(Request.class);
			Predicate condition = criteriaBuilder.equal(root.get(CommonEnum.REQUEST_CREATED_BY.getText()), uid);
			query.where(condition);
			listRequest = session.createQuery(query).getResultList();
		} catch (Exception ex) {
			LOGGER.error("Exception occurred while querying  getRequestsByDataManager.", ex);
			throw new GenisisDAOException("Exception occurred while querying  getRequestsByDataManager.", ex);
		}
		return listRequest;
	}

	/**
	 * This method is used to getRequestsDataSourceManager.
	 * 
	 * @param uid
	 *            The uid.
	 * @return this returns list of requests data source manager.
	 */
	@Override
	public List<Request> getRequestsDataSourceManagers(String uid) throws GenisisDAOException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("getRequestsDataSourceManagers: " + FortifyHelper.getCleanStringInput(uid));
		}
		List<Request> listRequest = null;
		try {
			
			Session session = sessionFactory.getCurrentSession();
			String select = StringUtils.EMPTY;
			// join
			select = select + " from Request request join User usr on request.createdBy = usr.username join UserRoleType userRoleType on usr.userId=userRoleType.userId  where ";
			// Request Type 1 query
			select = select + " ( request.requestTypeId=1 and (request.createdBy=:uid and request.statusDescription='" + WorkflowStatusEnum.DRAFT.getDesc() +"'  or request.statusDescription='" +  WorkflowStatusEnum.SENT.getDesc() + "'  or request.statusDescription='"+ WorkflowStatusEnum.REQUESTACCEPTED.getDesc() + "' or request.statusDescription='" + WorkflowStatusEnum.REQUESTNOTACCEPTED.getDesc() +"'))";
			select = select + " or ";
			// Request Type 2 query
			select = select + "(request.requestTypeId=2 and (request.createdBy= :uid or ( ( request.createdBy <> :uid and userRoleType.roleTypeId <> 4 and  request.statusDescription<>'" + WorkflowStatusEnum.DRAFT.getDesc() + "') and  ((request.statusDescription='" +  WorkflowStatusEnum.CANCELLED.getDesc() +"' and request.modifiedBy= :uid ) or request.statusDescription='" +  WorkflowStatusEnum.RESULTSDELIVERED.getDesc() +"'  or request.statusDescription='" +  WorkflowStatusEnum.RESULTSACCEPTED.getDesc() + "' or request.statusDescription='" + WorkflowStatusEnum.RESULTSNOTACCEPTED.getDesc() +"'))))";
	
			// String select ="from Request request join User usr on request.createdBy = usr.username join UserRoleType userRoleType on usr.userId=userRoleType.userId where  request.requestTypeId=1 and (request.createdBy=:uid and request.statusDescription='Draft' or request.statusDescription='Returned' or request.statusDescription='Submitted' or request.statusDescription='Denied') or request.requestTypeId=2 and request.createdBy= :uid or ( ( request.createdBy <> :uid and userRoleType.roleTypeId <> 4 and  request.statusDescription<>'Draft') and  (request.statusDescription='Cancelled' or request.statusDescription='ResultsDelivered' or request.statusDescription='ResultsNotAccepted' or request.statusDescription='ResultsAccepted')) ";
			LOGGER.info("getRequestsDataSourceManagers  Query executed: " + select);
			
			Query query = session.createQuery(select);
			query.setParameter("uid", uid);
			List<?> result = query.getResultList();
			listRequest = (List<Request>) result;
			
			
			
		/*	Session session = sessionFactory.getCurrentSession();
			CriteriaBuilder criteriaBuilder = session.getCriteriaBuilder();
			CriteriaQuery<Request> query = criteriaBuilder.createQuery(Request.class);
			Root<Request> root = query.from(Request.class);
			Predicate condition01 = criteriaBuilder.equal(root.get(CommonEnum.REQUEST_CREATED_BY.getText()), uid);
			Predicate condition02 = criteriaBuilder.equal(root.get(CommonEnum.STATUS_DESC.getText()), WorkflowStatusEnum.DRAFT.getDesc());
			Predicate condition03 = criteriaBuilder.equal(root.get(CommonEnum.STATUS_DESC.getText()), WorkflowStatusEnum.RETURNED.getDesc());
			Predicate condition04 = criteriaBuilder.equal(root.get(CommonEnum.STATUS_DESC.getText()), WorkflowStatusEnum.DENIED.getDesc());
			Predicate condition05 = criteriaBuilder.equal(root.get(CommonEnum.STATUS_DESC.getText()), WorkflowStatusEnum.SUBMITTED.getDesc());
			
			Predicate condition1 = criteriaBuilder.and(condition01, condition02);
			Predicate condition6 = criteriaBuilder.and(condition01, condition03);
			Predicate condition7 = criteriaBuilder.and(condition01, condition04);
			Predicate condition8 = criteriaBuilder.and(condition01, condition05);
			
			Predicate condition2 = null;
			Predicate condition3 = null;
			Predicate condition4 = null;
			Predicate condition5 = null;
			Predicate conditionOR = null;
			Predicate conditionVINCI = null;
			
			Predicate conditionTransferVINCI = null;
			// Request Type =1 (VINCI Request)
			condition2 = criteriaBuilder.equal(root.get(CommonEnum.REQUEST_TYPE_ID.getText()), CommonEnum.VINCI_REQUEST.getText());
			condition3 = criteriaBuilder.equal(root.get(CommonEnum.STATUS_DESC.getText()), WorkflowStatusEnum.SENT.getDesc());
			condition4 = criteriaBuilder.equal(root.get(CommonEnum.STATUS_DESC.getText()), WorkflowStatusEnum.REQUESTACCEPTED.getDesc());
			condition5 = criteriaBuilder.equal(root.get(CommonEnum.STATUS_DESC.getText()), WorkflowStatusEnum.REQUESTNOTACCEPTED.getDesc());
			conditionOR = criteriaBuilder.or(condition1, condition3, condition4, condition5);
			conditionVINCI = criteriaBuilder.and(condition2, conditionOR);
			
			// Request Type =2 (VINCI Request)
			condition2 = criteriaBuilder.equal(root.get(CommonEnum.REQUEST_TYPE_ID.getText()), CommonEnum.TRANSFER_TO_VINCI.getText());
			condition3 = criteriaBuilder.equal(root.get(CommonEnum.STATUS_DESC.getText()), WorkflowStatusEnum.RESULTSDELIVERED.getDesc());
			condition4 = criteriaBuilder.equal(root.get(CommonEnum.STATUS_DESC.getText()), WorkflowStatusEnum.RESULTSACCEPTED.getDesc());
			condition5 = criteriaBuilder.equal(root.get(CommonEnum.STATUS_DESC.getText()), WorkflowStatusEnum.RESULTSNOTACCEPTED.getDesc());
			condition6 = criteriaBuilder.equal(root.get(CommonEnum.STATUS_DESC.getText()), WorkflowStatusEnum.CANCELLED.getDesc());
			
			Predicate conditionAND=null;
			// uid and createdby are same
			conditionOR = criteriaBuilder.or(condition3, condition4, condition5, condition6);
			conditionAND= criteriaBuilder.and(condition01, conditionOR);
			
			// uid and created by different
			Predicate condition10 = criteriaBuilder.notEqual(root.get(CommonEnum.REQUEST_CREATED_BY.getText()), uid);
			Predicate condition11=criteriaBuilder.and(condition10, conditionOR);
			Predicate condition12 = criteriaBuilder.or(conditionAND, condition11);
			
			conditionTransferVINCI = criteriaBuilder.and(condition2, condition12);
			
			Predicate conditions = criteriaBuilder.or(conditionVINCI, conditionTransferVINCI);
			query.where(conditions);
			listRequest = session.createQuery(query).getResultList();*/
		} catch (Exception ex) {
			LOGGER.error("Exception occurred while querying  getRequestsDataSourceManagers.", ex);
			throw new GenisisDAOException("Exception occurred while querying  getRequestsDataSourceManagers.", ex);
		}
		return listRequest;
	}

	/**
	 * This method is used to get request by id.
	 * 
	 * @param id
	 *            The id.
	 * @return Request This returns request.
	 */
	@Override
	public Request getRequestById(int id) throws GenisisDAOException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("getRequestsById: " + id);
		}

		Request request = null;
		try {
			Session session = sessionFactory.getCurrentSession();
			request = session.get(Request.class, id);
		} catch (Exception ex) {
			LOGGER.error("Exception occurred while querying getRequestsById.", ex);
			throw new GenisisDAOException("Exception occurred while querying getRequestsById.", ex);
		}
		return request;
	}

	/**
	 * This method is used to get all requests.
	 * 
	 * @return this returns list of request.
	 */
	@Override
	public List<Request> getAllRequests() throws GenisisDAOException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("getAllRequests: ");
		}

		List<Request> listRequest = null;
		try {
			Session session = sessionFactory.getCurrentSession();
			CriteriaBuilder criteriaBuilder = session.getCriteriaBuilder();
			CriteriaQuery<Request> query = criteriaBuilder.createQuery(Request.class);
			Root<Request> root = query.from(Request.class);
			query.select(root);
			listRequest = session.createQuery(query).getResultList();
		} catch (Exception ex) {
			LOGGER.error("Exception occurred while querying  getAllRequests.", ex);
			throw new GenisisDAOException("Exception occurred while querying  getAllRequests.", ex);
		}
		return listRequest;
	}

	/**
	 * This method is used to getAllRequestsByStudyApproval.
	 * 
	 * @param studyApprovalId
	 *            The studyApprovalId.
	 * @return this returns list of requests.
	 */
	@Override
	public List<Request> getAllRequestsByStudyApproval(int studyApprovalId) throws GenisisDAOException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("getAllRequestsByStudyApproval: " + studyApprovalId);
		}

		List<Request> listRequest = null;
		try {
			Session session = sessionFactory.getCurrentSession();
			CriteriaBuilder criteriaBuilder = session.getCriteriaBuilder();
			CriteriaQuery<Request> query = criteriaBuilder.createQuery(Request.class);
			Root<Request> root = query.from(Request.class);
			Predicate condition = criteriaBuilder.equal(root.get("studyApproval"), studyApprovalId);
			query.where(condition);
			listRequest = session.createQuery(query).getResultList();
		} catch (Exception ex) {
			LOGGER.error("Exception occurred while querying  getAllRequestsByStudyApproval.", ex);
			throw new GenisisDAOException("Exception occurred while querying  getAllRequestsByStudyApproval.", ex);
		}
		return listRequest;
	}

	/**
	 * This method is used to getAllRequestsByStaus.
	 * 
	 * @param status
	 *            The status.
	 * @return this returns list of requets.
	 */
	@Override
	public List<Request> getAllRequestsByStaus(String status) throws GenisisDAOException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("getAllRequestsByStaus: " + FortifyHelper.getCleanStringInput(status));
		}

		List<Request> listRequest = null;
		try {
			Session session = sessionFactory.getCurrentSession();
			CriteriaBuilder criteriaBuilder = session.getCriteriaBuilder();
			CriteriaQuery<Request> query = criteriaBuilder.createQuery(Request.class);
			Root<Request> root = query.from(Request.class);
			Predicate condition = criteriaBuilder.equal(root.get(CommonEnum.STATUS_DESC.getText()), status);
			query.where(condition);
			listRequest = session.createQuery(query).getResultList();
		} catch (Exception ex) {
			LOGGER.error("Exception occurred while querying  getAllRequestsByStaus.", ex);
			throw new GenisisDAOException("Exception occurred while querying  getAllRequestsByStaus.", ex);
		}
		return listRequest;
	}

	/**
	 * This method is used to submit Or Modify entity.
	 * 
	 * @param entity
	 *            The entity.
	 * @return int This returns entity's id.
	 */
	@Override
	public int submitOrModify(Request entity) throws GenisisDAOException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("submitOrModify (Request) ");
		}

		try {
			Session session = sessionFactory.getCurrentSession();
			session.update(entity);
			session.flush();
		} catch (Exception ex) {
			LOGGER.error("Exception occurred on submitOrModify Request.", ex);
			throw new GenisisDAOException("Exception occurred on submitOrModify Request.", ex);
		}
		Request request = this.getRequestById(entity.getId());
		return (null != request) ? request.getId() : null;
	}

	/**
	 * Max row value.
	 *
	 * @return int This returns entity's id.
	 */
	private int maxRowValue() {
		// TODO: createCriteria Deprecated
		int maxRowVal = 1;
		Integer result = 0;
		try {
			Session session = sessionFactory.getCurrentSession();
			@SuppressWarnings("deprecation")
			Criteria criteria = session.createCriteria(Request.class).setProjection(Projections.max("id"));
			result = (Integer) criteria.uniqueResult();
		} catch (Exception ex) {
			LOGGER.error("Exception occurred on maxRowValue().", ex);
		}
		if (result != null) {
			maxRowVal = result.intValue() + 1;
		}
		return maxRowVal;
	}

	/**
	 * This method is used to submitOrModify RequestHistory.
	 * 
	 * @param entity
	 *            The entity.
	 * @return int This returns RequestHistory id.
	 * 
	 */
	@Override
	public int submitOrModify(RequestHistory entity) throws GenisisDAOException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("submitOrModify Request History: ");
		}

		int historyLineNo;
		int maxRowHistory;
		if (StringUtils.equals(entity.getStatusDescription(), WorkflowStatusEnum.DRAFT.getDesc())) {
			historyLineNo = this.historyLineForRequestHistory(entity.getRequest().getId());
			if (historyLineNo > 0) {
				entity.setHistLineNo(historyLineNo);
			} else {
				maxRowHistory = maxRowValueHistory();
				entity.setHistLineNo(maxRowHistory);
			}
		} else {
			maxRowHistory = maxRowValueHistory();
			entity.setHistLineNo(maxRowHistory);
		}
		try {
			Session session = sessionFactory.getCurrentSession();
			session.saveOrUpdate(entity);
			session.flush();

			if (LOGGER.isInfoEnabled()) {
				LOGGER.info("Saved Request History!!");
			}
		} catch (Exception ex) {
			LOGGER.error("Exception occurred on submitOrModify Request History", ex);
			throw new GenisisDAOException("Exception occurred on submitOrModify Request History", ex);
		}
		return entity.getHistLineNo();
	}

	/**
	 * Max row value history.
	 *
	 * @return int This returns id.
	 */
	private int maxRowValueHistory() {
		// TODO: createCriteria Deprecated
		int maxRowVal = 1;
		Integer result = 0;
		try {
			Session session = sessionFactory.getCurrentSession();
			@SuppressWarnings("deprecation")
			Criteria criteria = session.createCriteria(RequestHistory.class).setProjection(Projections.max("id"));
			result = (Integer) criteria.uniqueResult();
		} catch (Exception ex) {
			LOGGER.error("Exception occurred on maxRowValue().", ex);
		}
		if (result != null) {
			maxRowVal = result.intValue() + 1;
		}
		return maxRowVal;
	}

	/**
	 * This method is used to historyLineForRequestHistory.
	 *
	 * @param requestId
	 *            The requestid.
	 * @return int This returns requestId.
	 */
	private int historyLineForRequestHistory(int requestId) {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("historyLineForRequestHistory: ");
		}

		RequestHistory result = null;
		try {
			Session session = sessionFactory.getCurrentSession();
			CriteriaBuilder criteriaBuilder = session.getCriteriaBuilder();
			CriteriaQuery<RequestHistory> query = criteriaBuilder.createQuery(RequestHistory.class);
			Root<RequestHistory> root = query.from(RequestHistory.class);
			Predicate condition1 = criteriaBuilder.equal(root.get(CommonEnum.REQUEST_ENTITY.getText()).get("id"), requestId);
			Predicate condition2 = criteriaBuilder.equal(root.get(CommonEnum.STATUS_DESC.getText()), WorkflowStatusEnum.DRAFT.getDesc());
			Predicate conditions = criteriaBuilder.and(condition1, condition2);
			query.where(conditions);
			result = session.createQuery(query).getSingleResult();
		} catch (NoResultException e) {
			LOGGER.info(ErrorEnum.USER_DAO_EXP_GETUSERDETAILSBYUSERNAME.getErrorMessage());
			LOGGER.info(e.getMessage());
		} catch (Exception ex) {
			LOGGER.error("Exception occurred while querying  historyLineForRequestHistory.", ex);
		}

		int histLineNo = 0;
		if (result != null) {
			histLineNo = result.getHistLineNo();
		}
		String logInfo = "Request Id : " + requestId + "  Draft present in RequestHistory. HistLineNo: " + histLineNo;

		if (LOGGER.isInfoEnabled()) {
			LOGGER.info(logInfo);
		}

		return histLineNo;
	}

	/**
	 * This method is used to getRequestHistoryById.
	 * 
	 * @param requestId
	 *            The requestId.
	 * @return this returns list of requests.
	 */
	@Override
	public List<RequestHistory> getRequestHistoriesById(int requestId) throws GenisisDAOException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("getRequestHistoryById: " + requestId);
		}

		List<RequestHistory> listRequestHistory = null;
		try {
			Session session = sessionFactory.getCurrentSession();
			CriteriaBuilder criteriaBuilder = session.getCriteriaBuilder();
			CriteriaQuery<RequestHistory> query = criteriaBuilder.createQuery(RequestHistory.class);
			Root<RequestHistory> root = query.from(RequestHistory.class);
			Predicate condition = criteriaBuilder.equal(root.get(CommonEnum.REQUEST_ENTITY.getText()), requestId);
			query.where(condition);
			listRequestHistory = session.createQuery(query).getResultList();
		} catch (Exception ex) {
			LOGGER.error("Exception occurred while querying  getRequestHistoryById.", ex);
			throw new GenisisDAOException("Exception occurred while querying  getRequestHistoryById.", ex);
		}
		return listRequestHistory;
	}

	@Override
	public List<RequestHistory> getRequestHistoriesByRequestIdAndStatus(int requestId, String status) throws GenisisDAOException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("getRequestHistoryByRequestIdAndStatus: " + "Request Id: " + requestId + " Status:" + status);
		}

		List<RequestHistory> requestHistory = null;
		try {
			Session session = sessionFactory.getCurrentSession();
			CriteriaBuilder criteriaBuilder = session.getCriteriaBuilder();
			CriteriaQuery<RequestHistory> query = criteriaBuilder.createQuery(RequestHistory.class);
			Root<RequestHistory> root = query.from(RequestHistory.class);
			Predicate condition1 = criteriaBuilder.equal(root.get(CommonEnum.REQUEST_ENTITY.getText()).get("id"), requestId);
			Predicate condition2 = criteriaBuilder.equal(root.get(CommonEnum.STATUS_DESC.getText()), status);
			Predicate conditions = criteriaBuilder.and(condition1, condition2);
			query.where(conditions);
			requestHistory = session.createQuery(query).getResultList();
		} catch (Exception ex) {
			LOGGER.error("Exception occurred while querying  getRequestHistoryByRequestIdAndStatus.", ex);
			throw new GenisisDAOException("Exception occurred while querying  getRequestHistoryByRequestIdAndStatus.", ex);
		}
		return requestHistory;
	}

	@SuppressWarnings({ "deprecation", "unchecked", "rawtypes" })
	@Override
	public List<RequestHistory> getRequestHistoriesWithCommentsById(int requestId) throws GenisisDAOException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("getRequestHistoriesWithCommentsById: " + requestId);
		}

		List<RequestHistory> listRequestHistory = null;
		try {
			Session session = sessionFactory.getCurrentSession();
		//	String select = "FROM RequestHistory  requestHistoryTable left JOIN CommentHistory commentHistoryTable ON requestHistoryTable.taskId=commentHistoryTable.taskId  and requestHistoryTable.request.id=commentHistoryTable.request.id  where requestHistoryTable.id = :requestID order by requestHistoryTable.histLineNo ,commentHistoryTable.type desc, commentHistoryTable.commentId";
		//	String select = "FROM RequestHistory  requestHistoryTable left JOIN CommentHistory commentHistoryTable ON requestHistoryTable.taskId=commentHistoryTable.taskId  and requestHistoryTable.request.id=commentHistoryTable.request.id  where requestHistoryTable.request.id = :requestID order by requestHistoryTable.histLineNo ,commentHistoryTable.type desc, commentHistoryTable.commentId";
			
			String select = "FROM RequestHistory  requestHistoryTable left JOIN CommentHistory commentHistoryTable ON requestHistoryTable.taskId=commentHistoryTable.taskId  and requestHistoryTable.request.id=commentHistoryTable.request.id  where requestHistoryTable.request.id = :requestID  order by requestHistoryTable.histLineNo ,commentHistoryTable.type desc, commentHistoryTable.commentId";
			
				
			Query query = session.createQuery(select);
			query.setParameter("requestID", requestId);
			List<?> result = query.getResultList();
			listRequestHistory = (List<RequestHistory>) result;
		} catch (Exception ex) {
			LOGGER.error("Exception occurred while querying  getRequestHistoryById.", ex);
			throw new GenisisDAOException("Exception occurred while querying  getRequestHistoryById.", ex);
		}

		return listRequestHistory;
	}

	@Override
	public List<WorkflowStatus> getRequestStatusCounts() throws GenisisDAOException {
		List<WorkflowStatus> requestStatusCounts = null;
		try {
			Session session = sessionFactory.getCurrentSession();
			EntityManagerFactory entityManagerFactory = session.getEntityManagerFactory();
			CriteriaBuilder criteriaBuilder = entityManagerFactory.getCriteriaBuilder();
			CriteriaQuery<WorkflowStatus> criteriaQuery = criteriaBuilder.createQuery(WorkflowStatus.class);
			Root<WorkflowStatus> root = criteriaQuery.from(entityManagerFactory.getMetamodel().entity(WorkflowStatus.class));
			criteriaQuery.select(root);

			Subquery<Long> subquery = criteriaQuery.subquery(Long.class);
			Root<Request> rootRequest = subquery.from(Request.class);
			subquery.select(criteriaBuilder.count(rootRequest.get(CommonEnum.STATUS_ID.getText())));
			Predicate condition = criteriaBuilder.equal(rootRequest.get(CommonEnum.STATUS_DESC.getText()), root.get(CommonEnum.STATUS_WORKFLOW_DESC.getText()));
			subquery.where(condition);
			subquery.groupBy(rootRequest.get(CommonEnum.STATUS_DESC.getText()));

			criteriaQuery.select(criteriaBuilder.construct(WorkflowStatus.class, root.get(CommonEnum.STATUS_ID.getText()), root.get(CommonEnum.STATUS_WORKFLOW_DESC.getText()), subquery.getSelection()));
			criteriaQuery.groupBy(root.get(CommonEnum.STATUS_WORKFLOW_DESC.getText()), root.get(CommonEnum.STATUS_ID.getText()));
			requestStatusCounts = session.createQuery(criteriaQuery).getResultList();
		} catch (NoResultException e) {
			// Swallow exception
			LOGGER.error(ErrorEnum.REQUEST_DAO_EXP_GETREQUESTSTATUSCOUNTS.getErrorMessage(), e);
		} catch (Exception ex) {
			LOGGER.error(ErrorEnum.REQUEST_DAO_EXP_GETREQUESTSTATUSCOUNTS.getErrorMessage(), ex);
			throw new GenisisDAOException(ex.getClass().getSimpleName(), ex);
		}
		return requestStatusCounts;
	}

	@Override
	public WorkflowStatus getWorkflowStatus(String status) throws GenisisDAOException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("getWorkflowStatus: " + status);
		}

		WorkflowStatus workflowStatus = null;
		try {
			Session session = sessionFactory.getCurrentSession();
			CriteriaBuilder criteriaBuilder = session.getCriteriaBuilder();
			CriteriaQuery<WorkflowStatus> query = criteriaBuilder.createQuery(WorkflowStatus.class);
			Root<WorkflowStatus> root = query.from(WorkflowStatus.class);
			Predicate condition = criteriaBuilder.equal(root.get(CommonEnum.STATUS_WORKFLOW_DESC.getText()), status);
			query.where(condition);
			workflowStatus = session.createQuery(query).getSingleResult();
		} catch (NoResultException e) {
			LOGGER.info(ErrorEnum.REQUEST_DAO_EXP_GETWORKFLOWSTATUS.getErrorMessage());
			LOGGER.info(e.getMessage());
		} catch (Exception ex) {
			LOGGER.error(ErrorEnum.REQUEST_DAO_EXP_GETWORKFLOWSTATUS.getErrorMessage(), ex);
			throw new GenisisDAOException("Exception occurred while querying  getWorkflowStatus.", ex);
		}
		return workflowStatus;
	}
	
	


}